#![allow(clippy::result_large_err)]

use std::path::PathBuf;

use clap::Parser;
use clingwrap::config::ConfigLoader;
use directories::ProjectDirs;
use log::{debug, info, LevelFilter};

use ambient_ci::config::{Config, ConfigError, StoredConfig};

mod cmd;
use cmd::{AmbientDriverError, Leaf};

const QUAL: &str = "liw.fi";
const ORG: &str = "Ambient CI";
const APP: &str = env!("CARGO_BIN_NAME");

fn main() {
    if let Err(e) = fallible_main() {
        eprintln!("ERROR: {e}");
        let mut source = e.source();
        while let Some(src) = source {
            eprintln!("caused by: {src}");
            source = src.source();
        }
        std::process::exit(1);
    }
}

fn fallible_main() -> Result<(), Box<dyn std::error::Error>> {
    let args = Args::parse();
    setup_env_logger("AMBIENT_LOG", LevelFilter::Info);
    info!("{APP} starts");
    let config = args.config()?;
    match &args.cmd {
        Command::Actions(x) => x.run(&config)?,
        Command::Config(x) => x.run(&config)?,
        Command::Image(x) => x.run(&config)?,
        Command::Log(x) => x.run(&config)?,
        Command::Plan(x) => x.run(&config)?,
        Command::Projects(x) => x.run(&config)?,
        Command::Qemu(x) => x.run(&config)?,
        Command::Run(x) => x.run(&config)?,
    }
    info!("{APP} ends successfully");
    Ok(())
}

fn setup_env_logger(var_name: &str, level_filter: LevelFilter) {
    if std::env::var(var_name).is_ok() {
        env_logger::init_from_env(var_name);
    } else {
        env_logger::builder().filter_level(level_filter).init();
    }
}

#[derive(Debug, Parser)]
#[clap(name = APP, version = env!("VERSION"))]
pub struct Args {
    /// Configuration files to use in addition of the default one,
    /// unless `--no-config` is used.
    #[clap(long)]
    config: Vec<PathBuf>,

    /// Don't load default configuration file, but do load any files
    /// specified with `--config`.
    #[clap(long)]
    no_config: bool,

    /// Operation
    #[clap(subcommand)]
    cmd: Command,
}

impl Args {
    fn config(&self) -> Result<Config, ConfigError> {
        let mut loader = ConfigLoader::default();
        if self.no_config {
            debug!("--no-config used, using built-in default configuration");
        } else {
            let dirs = ProjectDirs::from(QUAL, ORG, APP).ok_or(ConfigError::ProjectDirs)?;
            let filename = dirs.config_dir().join("config.yaml");
            debug!(
                "load default configuration file {} if it exists",
                filename.display()
            );
            loader.allow_yaml(&filename);
        }

        for filename in self.config.iter() {
            debug!("load files named with --config: {}", filename.display());
            loader.require_yaml(filename);
        }

        let validator = StoredConfig::default();
        let config = loader
            .load(None, None, &validator)
            .map_err(ConfigError::Load)?;

        debug!("complete configuration: {config:#?}");
        Ok(config)
    }
}

#[derive(Debug, Parser)]
enum Command {
    Actions(cmd::actions::Actions),
    Image(ImageCmd),
    Config(cmd::config::ConfigCmd),
    Log(cmd::log::Log),
    Plan(cmd::plan::Plan),
    Projects(cmd::projects::ProjectsCmd),
    Qemu(cmd::qemu::QemuCmd),
    Run(cmd::run::Run),
}

#[derive(Debug, Parser)]
pub struct ImageCmd {
    #[clap(subcommand)]
    cmd: ImageSubCmd,
}

impl ImageCmd {
    fn run(&self, config: &Config) -> Result<(), AmbientDriverError> {
        match &self.cmd {
            ImageSubCmd::CloudInit(x) => x.run(config)?,
            ImageSubCmd::Import(x) => x.run(config)?,
            ImageSubCmd::List(x) => x.run(config)?,
            ImageSubCmd::Prepare(x) => x.run(config)?,
            ImageSubCmd::Remove(x) => x.run(config)?,
            ImageSubCmd::Show(x) => x.run(config)?,
            ImageSubCmd::Verify(x) => x.run(config)?,
        }
        Ok(())
    }
}

#[derive(Debug, Parser)]
enum ImageSubCmd {
    CloudInit(cmd::image::CloudInit),
    Import(cmd::image::ImportImage),
    List(cmd::image::ListImages),
    Prepare(cmd::image::PrepareImage),
    Remove(cmd::image::RemoveImages),
    Show(cmd::image::ShowImage),
    Verify(cmd::image::VerifyImage),
}
